# Copyright 2020 Logic and Optimization Group, Universitat de Lleida. All Rights Reserved.
from datetime import datetime, timedelta, timezone
from zipfile import ZipFile, ZIP_STORED
import pathlib
import hashlib
import shutil
import json
import uuid
import re
from flask import current_app, redirect
from wsDatabase import get_db

def validate_date(date_match_obj):
    try:
        utc_offset = timezone(
            timedelta() if date_match_obj.group('offset') == 'Z' else \
            timedelta(
                hours=int(date_match_obj.group('offset_hour')) * (1 if date_match_obj.group('offset_sign') == '+' else -1),
                minutes=int(date_match_obj.group('offset_minute'))
            )
        )
        date_datetime = datetime(
            year=int(date_match_obj.group('year')),
            month=int(date_match_obj.group('month')),
            day=int(date_match_obj.group('day')),
            hour=int(date_match_obj.group('hour')),
            minute=int(date_match_obj.group('minute')),
            second=int(date_match_obj.group('second')),
            tzinfo=utc_offset
        )
        return date_datetime
    except ValueError:
        return None

#########################################################################################

def enqueue_request(**params):
    # Validate params
    has_errors = {}
    iso8601_regex = r'^(?P<year>[0-9]{4})-?(?P<month>[0-9]{2})-?(?P<day>[0-9]{2})(T|\s)?(?P<hour>[0-9]{2}):?(?P<minute>[0-9]{2}):?(?P<second>[0-9]{2})\s?(?P<offset>Z|(?P<offset_sign>[+|-])(?P<offset_hour>[0-9]{2}):?(?P<offset_minute>[0-9]{2}))$'

    # from_date   date-time
    from_date_match = re.fullmatch(iso8601_regex, params['from_date'].upper())
    from_date_datetime = None
    if not from_date_match:
        has_errors['from_date'] = 'Date is not ISO8601 conforming.'
    else:
        from_date_datetime = validate_date(from_date_match)
        if not from_date_datetime:
            has_errors['from_date'] = 'Date is not valid.'

    # to_date     date-time
    to_date_match = re.fullmatch(iso8601_regex, params['to_date'].upper())
    to_date_datetime = None
    if not to_date_match:
        has_errors['to_date'] = 'Date is not ISO8601 conforming.'
    else:
        to_date_datetime = validate_date(to_date_match)
        if not to_date_datetime:
            has_errors['to_date'] = 'Date is not valid.'

    if from_date_datetime and to_date_datetime and (not from_date_datetime < to_date_datetime):
        has_errors['from_date'] = 'from_date should precede to_date.'
        has_errors['to_date'] = 'from_date should precede to_date.'

    # foi         file
    try:
        # should validate foi is really a GeoJSON file, for now just validating it is a JSON
        foi_json = json.load(params['foi'])#.read()
        foi_str = json.dumps(foi_json)
    except json.JSONDecodeError:
        has_errors['foi'] = 'foi is not a valid JSON file. Particularly, it should be a GeoJSON.'

    # epsg        string
    epsg_regex = r'^EPSG:(?P<epsg_code>[0-9]{4,})$'
    epsg_match = re.fullmatch(epsg_regex, params['epsg'])
    if not epsg_match:
        has_errors['epsg'] = 'EPSG is not a valid. It should have a format EPSG:numeric_code, where numeric_code is in the range [1024, 32767].'
    else:
        epsg_code = int(epsg_match.group('epsg_code'))
        if epsg_code < 1024 or epsg_code > 32767: # EPSG has a range of [1024, 32767]
            has_errors['epsg'] = 'EPSG code is not a valid EPSG code. It should be in the range [1024, 32767].'

    # cloud_cover  int
    cloud_cover = params['cloud_cover']
    if cloud_cover < 1 or 100 < cloud_cover:
        has_errors['cloud_cover'] = 'Cloud cover is the maximum admissible cloud cover percentage and should be in the range [1, 100].'

    # proc_cloud_probability int
    proc_cloud_probability = params['proc_mask_probability']
    if proc_cloud_probability < 1 or 100 < proc_cloud_probability:
        has_errors['proc_mask_probability'] = 'proc_mask_probability is the minimum percentage for a pixel to be declared as masked (from the cloud mask) and should be in the range [1, 100].'
    # proc_min_pix_cover int
    proc_min_pix_cover = params['proc_min_pix_cover']
    if proc_min_pix_cover < 0 or 100 < proc_min_pix_cover:
        has_errors['proc_min_pix_cover'] = 'proc_min_pix_cover is the minimum percentage of pixels that have to be cleared (not masked) to accept the image and should be in the range [0, 100].'

    # callback    string
    has_CB = False
    if params['callback'] != 'None':
        http_regex = r'https?:\/{2}.*$'
        callback_match = re.fullmatch(http_regex, params['callback'])
        if not callback_match:
            has_errors['callback'] = 'Callback is not an http link or None.'
        has_CB = True

    if len(has_errors) != 0:
        return {'err':'parameter errors', **has_errors}, 400

    # Create insert sql statement
    cb_par_str = ', Callback' if has_CB else ''
    cb_val_str = f", '{params['callback']}'" if has_CB else ''
    insert_query_params = f"INSERT INTO Queue (RequestID, IniDate, EndDate, Foi, Epsg, Cloud_Cover, Proc_MaskProb, Proc_MinPixCov{cb_par_str}) VALUES ("
    insert_query_values = f", '{params['from_date'].upper()}', '{params['to_date'].upper()}', '{foi_str}'::json,'{params['epsg']}','{params['cloud_cover']}','{params['proc_mask_probability']}','{params['proc_min_pix_cover']}'{cb_val_str})"
    sha1 = hashlib.sha1()
    sha1.update(insert_query_values.encode())
    sha1.update(str(uuid.uuid4()).encode())
    uid = uuid.uuid5(uuid.NAMESPACE_DNS, sha1.hexdigest())
    final_insert_query = insert_query_params + f"'{uid}'" + insert_query_values

    cur = get_db().cursor()
    cur.execute(final_insert_query)
    get_db().commit()

    return {'id': uid}, 200

def enqueue_download(**params):
    # Validate params
    has_errors = {}
    iso8601_regex = r'^(?P<year>[0-9]{4})-?(?P<month>[0-9]{2})-?(?P<day>[0-9]{2})(T|\s)?(?P<hour>[0-9]{2}):?(?P<minute>[0-9]{2}):?(?P<second>[0-9]{2})\s?(?P<offset>Z|(?P<offset_sign>[+|-])(?P<offset_hour>[0-9]{2}):?(?P<offset_minute>[0-9]{2}))$'

    # from_date   date-time
    from_date_match = re.fullmatch(iso8601_regex, params['from_date'].upper())
    from_date_datetime = None
    if not from_date_match:
        has_errors['from_date'] = 'Date is not ISO8601 conforming.'
    else:
        from_date_datetime = validate_date(from_date_match)
        if not from_date_datetime:
            has_errors['from_date'] = 'Date is not valid.'

    # to_date     date-time
    to_date_match = re.fullmatch(iso8601_regex, params['to_date'].upper())
    to_date_datetime = None
    if not to_date_match:
        has_errors['to_date'] = 'Date is not ISO8601 conforming.'
    else:
        to_date_datetime = validate_date(to_date_match)
        if not to_date_datetime:
            has_errors['to_date'] = 'Date is not valid.'

    if from_date_datetime and to_date_datetime and (not from_date_datetime < to_date_datetime):
        has_errors['from_date'] = 'from_date should precede to_date.'
        has_errors['to_date'] = 'from_date should precede to_date.'

    # foi         file
    try:
        # should validate foi is really a GeoJSON file, for now just validating it is a JSON
        foi_json = json.load(params['foi'])#.read()
        foi_str = json.dumps(foi_json)
    except json.JSONDecodeError:
        has_errors['foi'] = 'foi is not a valid JSON file. Particularly, it should be a GeoJSON.'

    # epsg        string
    epsg_regex = r'^EPSG:(?P<epsg_code>[0-9]{4,})$'
    epsg_match = re.fullmatch(epsg_regex, params['epsg'])
    if not epsg_match:
        has_errors['epsg'] = 'EPSG is not a valid. It should have a format EPSG:numeric_code, where numeric_code is in the range [1024, 32767].'
    else:
        epsg_code = int(epsg_match.group('epsg_code'))
        if epsg_code < 1024 or epsg_code > 32767: # EPSG has a range of [1024, 32767]
            has_errors['epsg'] = 'EPSG code is not a valid EPSG code. It should be in the range [1024, 32767].'

    # cloud_cover  int
    cloud_cover = params['cloud_cover']
    if cloud_cover < 1 or 100 < cloud_cover:
        has_errors['cloud_cover'] = 'Cloud cover is the maximum admissible cloud cover percentage and should be in the range [1, 100].'

    # callback    string
    has_CB = False
    if params['callback'] != 'None':
        http_regex = r'https?:\/{2}.*$'
        callback_match = re.fullmatch(http_regex, params['callback'])
        if not callback_match:
            has_errors['callback'] = 'Callback is not an http link or None.'
        has_CB = True

    if len(has_errors) != 0:
        return {'err':'parameter errors', **has_errors}, 400

    # Create insert sql statement
    cb_par_str = ', Callback' if has_CB else ''
    cb_val_str = f", '{params['callback']}'" if has_CB else ''
    insert_query_params = f"INSERT INTO Queue (RequestID, IniDate, EndDate, Foi, Epsg, Cloud_Cover, Req_Down{cb_par_str}) VALUES ("
    insert_query_values = f", '{params['from_date'].upper()}', '{params['to_date'].upper()}', '{foi_str}'::json,'{params['epsg']}','{params['cloud_cover']}','1'{cb_val_str})"
    sha1 = hashlib.sha1()
    sha1.update(insert_query_values.encode())
    sha1.update(str(uuid.uuid4()).encode())
    uid = uuid.uuid5(uuid.NAMESPACE_DNS, sha1.hexdigest())
    final_insert_query = insert_query_params + f"'{uid}'" + insert_query_values

    cur = get_db().cursor()
    cur.execute(final_insert_query)
    get_db().commit()

    return {'status': 'Download enqueued'}, 200

def request_status(**params):
    try:
        uuid.UUID(f'{params["id"]}')
    except ValueError:
        return {'err':'Malformed ID', **params}, 400

    cur = get_db().cursor()
    cur.execute(f"SELECT ReqState FROM Queue WHERE RequestID='{params['id']}'")
    dat = cur.fetchone()
    if dat is None:
        return {'err':'Not Found', **params}, 404

    dat = (dat[0].replace('_HALT', ''), )
    #'DOWN_TODO', 'DOWN_INPROC'
    if dat[0] == 'DOWN_TODO':
        dat = ('ENQUEUED', )
    elif dat[0] == 'DOWN_INPROC':
        dat = ('IN_PROCESS', )

    return {'status' : dat[0]}, 200

def delete_request(**params):
    try:
        uuid.UUID(f'{params["id"]}')
    except ValueError:
        return {'err':'Malformed ID', **params}, 400

    if 'SAT_PROC_DIR' not in current_app.config:
        return {'err':'Processed Image Directory not setup. Contact your admin.'}, 503

    cur = get_db().cursor()
    cur.execute(f"SELECT ReqState FROM Queue WHERE RequestID='{params['id']}'")
    dat = cur.fetchone()
    if dat is None:
        return {'err':'Not Found', **params}, 404

    if dat[0] == 'IN_PROCESS' or dat[0] == 'DOWN_INPROC':
        return {'err':'In process', **params}, 403

    cur.execute(f"DELETE FROM Queue WHERE RequestID='{params['id']}'")
    get_db().commit()

    SAT_PROC_DIR_REQ = pathlib.Path(current_app.config['SAT_PROC_DIR']) / params['id']
    shutil.rmtree(SAT_PROC_DIR_REQ)

    return {'status':'Request deleted', **params}, 200

def get_result(**params):
    res_dic, res_code = request_status(**params)
    if res_code != 200:
        return res_dic, res_code
    elif res_dic['status'] != 'FINISHED':
        return {'err':'Not yet in Finished state', **params}, 400

    if 'SAT_PROC_DIR' not in current_app.config:
        return {'err':'Processed Image Directory not setup. Contact your admin.'}, 503

    SAT_PROC_DIR_REQ = pathlib.Path(current_app.config['SAT_PROC_DIR']) / params['id']

    if params['format'] == 'list':
        image_files = []
        for tiff in SAT_PROC_DIR_REQ.glob('**/*.tiff'):
            tiff_file = str(tiff.relative_to(SAT_PROC_DIR_REQ))
            image_files.append(f'/get_file/{params["id"]}/{tiff_file}')
        return {'items': image_files}, 200

    if params['format'] == 'zip':
        zip_file_version = SAT_PROC_DIR_REQ / f'{params["id"]}.zip'
        if not zip_file_version.exists():
            with ZipFile(zip_file_version, 'x') as zipped_files:
                for tiff in SAT_PROC_DIR_REQ.glob('**/*.tiff'):
                    zipped_files.write(tiff, arcname=tiff.relative_to(current_app.config['SAT_PROC_DIR']), compress_type=ZIP_STORED)

        return redirect(f'/get_file/{params["id"]}/{params["id"]}.zip', code=303)
