#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""Simple arithmetic optimisation example extracted from the Minizinc tutorial,
section 2.2.

Problem description:

A banana cake which takes 250g of self-raising flour, 2 mashed bananas, 75g
sugar and 100g of butter, and a chocolate cake which takes 200g of self-raising
flour, 75g of cocoa, 150g sugar and 150g of butter. We can sell a chocolate cake
for $4.50 and a banana cake for $4.00. And we have 4kg self-raising flour, 6
bananas, 2kg of sugar, 500g of butter and 500g of cocoa. The question is how
many of each sort of cake should we bake for the fete to maximise the profit

Minizinc tutorial url:
    http://www.minizinc.org/downloads/doc-latest/minizinc-tute.pdf
"""

import argparse
import sys


# Available ingredients
COCOA = 500
FLOUR = 4000
SUGAR = 2000
BUTTER = 500
BANANAS = 6

# Required ingredients per cake
B_CAKE_FLOUR = 250
B_CAKE_SUGAR = 75
B_CAKE_BUTTER = 100
B_CAKE_BANANAS = 2

C_CAKE_COCOA = 75
C_CAKE_FLOUR = 200
C_CAKE_SUGAR = 150
C_CALE_BUTTER = 150

# Profit for each cake
B_CAKE_PROFIT = 4
C_CAKE_PROFIT = 4.5


# Program main
###############################################################################

def main():
    """Cakes program main function.

    :return: 0 on success, any other value on error.
    """
    parser = argparse.ArgumentParser(
        formatter_class=argparse.ArgumentDefaultsHelpFormatter)

    parser.add_argument('-b', '--banana', type=int, required=True,
                        help="Number of banana cakes.")
    parser.add_argument('-c', '--chocolate', type=int, required=True,
                        help="Number of chocolate cakes.")

    args = parser.parse_args()
    remaining = remaining_ingredients(args.banana, args.chocolate)
    if all(x >= 0 for x in remaining):
        b_profit = args.banana * B_CAKE_PROFIT
        c_profit = args.chocolate * C_CAKE_PROFIT
        print("Banana cakes profit:", b_profit)
        print("Chocolate cakes profit:", c_profit)
        print("GGA SUCCESS", -(b_profit + c_profit))
    else:
        # missing = sum(x for x in remaining if x < 0)
        print("GGA CRASHED ", 0)

    return 0


def remaining_ingredients(num_banana, num_chocolate):
    """Tests if it is plausible to make the specified number of cakes of each
    type.

    :param int num_banana: Number of banana cakes to make.
    :param num_chocolate: Number of chocolate cakes to make.
    :return: True if it is plausible, False otherwise.
    :rtype: bool
    """
    cocoa = COCOA - (C_CAKE_COCOA * num_chocolate)
    flour = FLOUR - (B_CAKE_FLOUR * num_banana + C_CAKE_FLOUR + num_chocolate)
    sugar = SUGAR - (B_CAKE_SUGAR * num_banana + C_CAKE_SUGAR * num_chocolate)
    butter = BUTTER - (B_CAKE_BUTTER * num_banana
                       + C_CALE_BUTTER * num_chocolate)
    bananas = BANANAS - (B_CAKE_BANANAS * num_banana)

    return cocoa, flour, sugar, butter, bananas


# Program entry point
###############################################################################

if __name__ == '__main__':
    try:
        sys.exit(main())
    except Exception as err:
        print("Captured exception:", err, file=sys.stderr)
        print("GGA ABORT")
        sys.exit(1)
